<?php
session_start();

// Configurações do banco de dados (usando SQLite para simplicidade)
define('DB_FILE', 'admin_users.db');

// Função para conectar ao banco de dados
function getDB() {
    $db = new SQLite3(DB_FILE);
    
    // Criar tabela se não existir
    $db->exec("
        CREATE TABLE IF NOT EXISTS admins (
            id INTEGER PRIMARY KEY AUTOINCREMENT,
            username TEXT UNIQUE NOT NULL,
            password TEXT NOT NULL,
            domain TEXT NOT NULL,
            created_at DATETIME DEFAULT CURRENT_TIMESTAMP
        )
    ");
    
    return $db;
}

// Função para registrar um novo admin
function registerAdmin($username, $password, $domain) {
    $db = getDB();
    
    // Verificar se o usuário já existe
    $stmt = $db->prepare("SELECT id FROM admins WHERE username = :username");
    $stmt->bindValue(':username', $username, SQLITE3_TEXT);
    $result = $stmt->execute();
    
    if ($result->fetchArray()) {
        return ['success' => false, 'message' => 'Usuário já existe'];
    }
    
    // Hash da senha
    $hashedPassword = password_hash($password, PASSWORD_DEFAULT);
    
    // Inserir novo admin
    $stmt = $db->prepare("
        INSERT INTO admins (username, password, domain) 
        VALUES (:username, :password, :domain)
    ");
    $stmt->bindValue(':username', $username, SQLITE3_TEXT);
    $stmt->bindValue(':password', $hashedPassword, SQLITE3_TEXT);
    $stmt->bindValue(':domain', $domain, SQLITE3_TEXT);
    
    if ($stmt->execute()) {
        return ['success' => true, 'message' => 'Administrador cadastrado com sucesso'];
    } else {
        return ['success' => false, 'message' => 'Erro ao cadastrar administrador'];
    }
}

// Função para fazer login
function login($username, $password) {
    $db = getDB();
    
    $stmt = $db->prepare("SELECT id, username, password, domain FROM admins WHERE username = :username");
    $stmt->bindValue(':username', $username, SQLITE3_TEXT);
    $result = $stmt->execute();
    $user = $result->fetchArray(SQLITE3_ASSOC);
    
    if ($user && password_verify($password, $user['password'])) {
        $_SESSION['admin_id'] = $user['id'];
        $_SESSION['admin_username'] = $user['username'];
        $_SESSION['admin_domain'] = $user['domain'];
        
        return ['success' => true, 'message' => 'Login realizado com sucesso'];
    } else {
        return ['success' => false, 'message' => 'Usuário ou senha inválidos'];
    }
}

// Função para verificar se está logado
function isLoggedIn() {
    return isset($_SESSION['admin_id']);
}

// Função para obter domínio do admin logado
function getAdminDomain() {
    return isset($_SESSION['admin_domain']) ? $_SESSION['admin_domain'] : null;
}

// Função para fazer logout
function logout() {
    session_destroy();
    return ['success' => true, 'message' => 'Logout realizado com sucesso'];
}

// Processar requisições
header('Content-Type: application/json');

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $action = $_POST['action'] ?? '';
    
    switch ($action) {
        case 'register_admin':
            $username = $_POST['username'] ?? '';
            $password = $_POST['password'] ?? '';
            $domain = $_POST['domain'] ?? '';
            
            if (empty($username) || empty($password) || empty($domain)) {
                echo json_encode(['success' => false, 'message' => 'Todos os campos são obrigatórios']);
                exit;
            }
            
            $result = registerAdmin($username, $password, $domain);
            echo json_encode($result);
            break;
            
        case 'login':
            $username = $_POST['username'] ?? '';
            $password = $_POST['password'] ?? '';
            
            if (empty($username) || empty($password)) {
                echo json_encode(['success' => false, 'message' => 'Usuário e senha são obrigatórios']);
                exit;
            }
            
            $result = login($username, $password);
            echo json_encode($result);
            break;
            
        case 'logout':
            $result = logout();
            echo json_encode($result);
            break;
            
        case 'check_session':
            echo json_encode(['logged_in' => isLoggedIn(), 'domain' => getAdminDomain()]);
            break;
            
        default:
            echo json_encode(['success' => false, 'message' => 'Ação não reconhecida']);
    }
} else {
    echo json_encode(['success' => false, 'message' => 'Método não permitido']);
}
?>